/*!
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2022-05-25
 *
 * @attention
 *
 *  Copyright (C) 2021-2022 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be usefull and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

#include "main.h"
#include "Board.h"
#include "stdio.h"
#include "apm32f4xx_gpio.h"
#include "apm32f4xx_adc.h"
#include "apm32f4xx_misc.h"
#include "apm32f4xx_usart.h"
#include "bsp_lcd.h"
#include "bsp_lcdFont.h"
#include "picture.h"

/** printf using USART1  */
#define DEBUG_USART  USART1

uint32_t i = 0;
uint32_t status = 0;

/** Delay */
void Delay(uint32_t count);
void LED_Demo(void);
void LCD_Demo(void);

/*!
 * @brief       Main program
 *
 * @param       None
 *
 * @retval      None
 */
int main(void)
{
    /** Configure LED GPIO Pin */
    APM_EVAL_LEDInit(LED1);
    APM_EVAL_LEDInit(LED2);
    APM_EVAL_LEDInit(LED3);

    APM_EVAL_PBInit(BUTTON_KEY1, BUTTON_MODE_GPIO);

    /** USART Initialization */
    USART_Config_T usartConfigStruct;

    /** USART configuration */
    USART_ConfigStructInit(&usartConfigStruct);
    usartConfigStruct.baudRate = 115200;
    usartConfigStruct.mode = USART_MODE_TX_RX;
    usartConfigStruct.parity = USART_PARITY_NONE;
    usartConfigStruct.stopBits = USART_STOP_BIT_1;
    usartConfigStruct.wordLength = USART_WORD_LEN_8B;
    usartConfigStruct.hardwareFlow = USART_HARDWARE_FLOW_NONE;
    /** COM1 init*/
    APM_EVAL_COMInit(COM1, &usartConfigStruct);

    LCD_GPIO_Init();
    LCD_SMC_Init();
    LCD_Reset();
    LCD_ConfigRegister();
    LCD_EnableBackLight();

    LCD_ConfigScan(LCD_SCAN_MODE_6);
    LCD_ConfigBackColor(LCD_COLOR_WHITE);
    LCD_ClearScreen();

    LCD_Demo();

    while (1)
    {
    }
}

void LCD_Demo(void)
{
    /** Display Geehy logo */
    LCD_DisplayPic(120, 1, 240, 108, imageGeehyLogo);
    LCD_PrintString(185, 90, "LCD Demo", LCD_COLOR_USER);
    LCD_DrawLine(10, 120, 470, 120, LCD_COLOR_USER);
    LCD_DrawRectangle(10, 130, LCD_ReadMaxLenX() - 10, LCD_ReadMaxLenY() - 10, LCD_COLOR_USER);

    while (1)
    {
        /** Open a window for demo */
        LCD_OpenWindow(11, 131, LCD_ReadMaxLenX() - 21, LCD_ReadMaxLenY() - 141);
        /** Clear window */
        LCD_FillWindowColor(LCD_ReadBackColor());

        switch(status)
        {
            case 0:
                LCD_DrawCircle(240, 300, 100, LCD_COLOR_RED, ENABLE);
                LCD_DrawCircle(360, 500, 100, LCD_COLOR_YELLOW, ENABLE);
                LCD_DrawCircle(120, 500, 100, LCD_COLOR_BLUE, ENABLE);
                APM_EVAL_LEDOn(LED1);       
                APM_EVAL_LEDOff(LED2);
                APM_EVAL_LEDOff(LED3);
                break;
            case 1:
                LCD_DrawRectangle(100, 400, 380, 500, LCD_COLOR_BLACK);
                LCD_DrawRectangle(180, 320, 300, 580, LCD_COLOR_RED);
                APM_EVAL_LEDOff(LED1);
                APM_EVAL_LEDOn(LED2);
                APM_EVAL_LEDOff(LED3);
                break;
            default:
                LCD_DrawLine(240, 300, 120, 500, LCD_COLOR_RED);
                LCD_DrawLine(240, 300, 360, 500, LCD_COLOR_YELLOW);
                LCD_DrawLine(120, 500, 360, 500, LCD_COLOR_BLUE);
                LCD_DrawLine(240, 550, 120, 350, LCD_COLOR_BLACK);
                LCD_DrawLine(240, 550, 360, 350, LCD_COLOR_USER);
                LCD_DrawLine(120, 350, 360, 350, LCD_COLOR_GREEN);
                LCD_DrawCircle(240, 425, 150, LCD_COLOR_RED, DISABLE);
                APM_EVAL_LEDOff(LED1);
                APM_EVAL_LEDOff(LED2);
                APM_EVAL_LEDOn(LED3);
                break;
        }
        while (1)
        {
            if (!APM_EVAL_PBGetState(BUTTON_KEY1))
            {
                while (!APM_EVAL_PBGetState(BUTTON_KEY1));
                
                if (status++ > 1)
                {
                    status = 0;
                }
                break;
                
               
            }
        }
    }
}

/*!
 * @brief       Delay
 *
 * @param       count:  delay count
 *
 * @retval      None
 */
void Delay(uint32_t count)
{
    volatile uint32_t delay = count;
    while(delay--);
}

/*!
 * @brief       Redirect C Library function printf to serial port.
 *              After Redirection, you can use printf function.
 *
 * @param       ch:  The characters that need to be send.
 *
 * @param       *f:  pointer to a FILE that can recording all information
 *              needed to control a stream
 *
 * @retval      The characters that need to be send.
 */
int fputc(int ch, FILE* f)
{
    /** send a byte of data to the serial port */
    USART_TxData(DEBUG_USART, (uint8_t)ch);

    /** wait for the data to be send  */
    while (USART_ReadStatusFlag(DEBUG_USART, USART_FLAG_TXBE) == RESET);

    return (ch);
}
